﻿using UnityEngine;
using System.Collections;
using System;
using UnityEngine.UI;

public interface IGuiUpdate
{
    void UpdateState();
}

public class GuiLogic : MonoBehaviour
{
	public Camera MainCamera;
	public Audio AudioSystem;

	public float BallSelectXRot = -25.0f;
	public float MainXRot = -30.0f;

	public float GetReadyTime = 1.0f;
	public float WinTime = 20.0f;
	public float ColourSpeed = 1.0f;
	public float RotateSpeed = 1.0f;
	public float IntroWaitTime = 2.0f;
	public float InitWaitTime = 1.0f;

	public GameObject StartButton;
	public GameObject RestartButton;
	public GameObject QuitButton;
    public GameObject MainGroup;
	public GameObject OptionsGroup;
	public GameObject ScoreGroup;
	public GameObject LevelSelect;
	public GameObject BallSelect;
	public GameObject BackPanel;
    public GameObject GetReadyGroup;
	public GameObject Blocks;
	public GameObject Logo;
	public GameObject WinGroup;
	public GameObject Demo;
	public GameObject LevelComplete;

	public bool SkipGetReady;

	public void Start()
	{
		mSceneLogic = FindObjectOfType<SceneLogic>();

		var canvas = GetComponent<Canvas>();
		canvas.enabled = !Boot.sHideGui;
	}

	public void OnClickStartGame()
	{
		switch (mState)
		{
			case States.Intro:
			case States.Demo:
				// go from demo -> main
				mState = States.Main;
				FindObjectOfType<SceneLogic>().StartLevel(SceneLogic.StateChangeReason.LevelSelect);
				break;

			case States.Main:
				// go from main -> restart
				mState = States.Restart;
				break;

			default:
				if (SkipGetReady)
				{
					// straight into OSD
					mState = States.OSD;

					if (mSceneLogic != null)
					{
						// start game
						mSceneLogic.Play();
					}
				}
				else
				{
					// get ready, then OSD
					mState = States.GetReady;
				}
				break;
		}

        UpdateState();
    }

	public void OnClickRestartGame()
	{
		OnClickStartGame();
/*
		if (mState == States.Restart)
		{
			mState = States.GetReady;
			UpdateState();
		}
*/
	}

	public void OnClickQuit()
    {
        mState = States.Main;
        UpdateState();
    }

	public void OnClickOptions()
	{
		mState = States.Options;
		UpdateState();
	}

	public void OnClickLevelSelect()
    {
        mState = States.LevelSelect;
        UpdateState();
    }

	public void OnClickBallSelect()
	{
		mState = States.BallSelect;
		UpdateState();
	}

	public void PlayerGuiSound()
	{
		var sfx = FindObjectOfType<SfxLogic>();
		sfx.PlaySfx(SfxLogic.Sfx.Gui);
	}

	public void UpdateState()
    {
        switch (mState)
        {
            case States.Main:
                StartButton.SetActive(true);
				RestartButton.SetActive(false);
				QuitButton.SetActive(false);
                MainGroup.SetActive(true);
                BackPanel.SetActive(true);
                ScoreGroup.SetActive(false);
                LevelSelect.SetActive(false);
				BallSelect.SetActive(false);
				GetReadyGroup.SetActive(false);
				Blocks.SetActive(true);
				Logo.SetActive(false);
				WinGroup.SetActive(false);
				Demo.SetActive(false);
				OptionsGroup.SetActive(false);
				LevelComplete.SetActive(false);

				// set main camera X angle
				MainCamera.transform.localRotation = Quaternion.Euler(MainXRot, 0, 0);
				break;

            case States.OSD:
                StartButton.SetActive(false);
				RestartButton.SetActive(false);
				QuitButton.SetActive(false);
                MainGroup.SetActive(false);
                BackPanel.SetActive(false);
                ScoreGroup.SetActive(false);
                LevelSelect.SetActive(false);
				BallSelect.SetActive(false);
				GetReadyGroup.SetActive(false);
				Blocks.SetActive(true);
				Logo.SetActive(false);
				WinGroup.SetActive(false);
				Demo.SetActive(false);
				OptionsGroup.SetActive(false);
				LevelComplete.SetActive(false);
				break;

			case States.Restart:
                StartButton.SetActive(true);
				RestartButton.SetActive(true);
				QuitButton.SetActive(true);
                MainGroup.SetActive(false);
                BackPanel.SetActive(false);
                ScoreGroup.SetActive(true);
                LevelSelect.SetActive(false);
				BallSelect.SetActive(false);
				GetReadyGroup.SetActive(false);
				Blocks.SetActive(true);
				Logo.SetActive(false);
				WinGroup.SetActive(false);
				Demo.SetActive(false);
				OptionsGroup.SetActive(false);
				LevelComplete.SetActive(false);

				// camera on
				MainCamera.gameObject.SetActive(true);
				break;

            case States.LevelSelect:
                StartButton.SetActive(false);
				RestartButton.SetActive(false);
				QuitButton.SetActive(true);
                MainGroup.SetActive(false);
                BackPanel.SetActive(true);
                ScoreGroup.SetActive(false);
                LevelSelect.SetActive(true);
				BallSelect.SetActive(false);
				GetReadyGroup.SetActive(false);
				Logo.SetActive(false);
				WinGroup.SetActive(false);
				Demo.SetActive(false);
				OptionsGroup.SetActive(false);
				LevelComplete.SetActive(false);
				break;

			case States.BallSelect:
				StartButton.SetActive(false);
				RestartButton.SetActive(false);
				QuitButton.SetActive(true);
				MainGroup.SetActive(false);
				BackPanel.SetActive(false);
				ScoreGroup.SetActive(false);
				LevelSelect.SetActive(false);
				BallSelect.SetActive(true);
				GetReadyGroup.SetActive(false);
				Blocks.SetActive(false);
				Logo.SetActive(false);
				WinGroup.SetActive(false);
				Demo.SetActive(false);
				OptionsGroup.SetActive(false);
				LevelComplete.SetActive(false);

				// set ball-select camera X angle
				MainCamera.transform.localRotation = Quaternion.Euler(BallSelectXRot, 0, 0);
				break;

			case States.GetReady:
                StartButton.SetActive(false);
				RestartButton.SetActive(false);
				QuitButton.SetActive(false);
                MainGroup.SetActive(false);
                BackPanel.SetActive(false);
                ScoreGroup.SetActive(false);
                LevelSelect.SetActive(false);
				BallSelect.SetActive(false);
				GetReadyGroup.SetActive(true);
				Blocks.SetActive(true);
				Logo.SetActive(false);
				WinGroup.SetActive(false);
				Demo.SetActive(false);
				OptionsGroup.SetActive(false);
				LevelComplete.SetActive(false);
				break;

			case States.Intro:
				StartButton.SetActive(true);
				RestartButton.SetActive(false);
				QuitButton.SetActive(false);
				MainGroup.SetActive(false);
				BackPanel.SetActive(true);
				ScoreGroup.SetActive(false);
				LevelSelect.SetActive(false);
				BallSelect.SetActive(false);
				GetReadyGroup.SetActive(false);
				Blocks.SetActive(true);
				Logo.SetActive(true);
				WinGroup.SetActive(false);
				Demo.SetActive(false);
				OptionsGroup.SetActive(false);
				LevelComplete.SetActive(false);
				break;

			case States.Demo:
				StartButton.SetActive(true);
				RestartButton.SetActive(false);
				QuitButton.SetActive(false);
				MainGroup.SetActive(false);
				BackPanel.SetActive(false);
				ScoreGroup.SetActive(false);
				LevelSelect.SetActive(false);
				BallSelect.SetActive(false);
				GetReadyGroup.SetActive(false);
				Blocks.SetActive(true);
				Logo.SetActive(false);
				WinGroup.SetActive(false);
				Demo.SetActive(true);
				OptionsGroup.SetActive(false);
				LevelComplete.SetActive(false);
				break;

			case States.Win:
				StartButton.SetActive(false);
				RestartButton.SetActive(false);
				QuitButton.SetActive(false);
				MainGroup.SetActive(false);
				BackPanel.SetActive(false);
				ScoreGroup.SetActive(false);
				LevelSelect.SetActive(false);
				BallSelect.SetActive(false);
				GetReadyGroup.SetActive(false);
				Blocks.SetActive(false);
				Logo.SetActive(false);
				WinGroup.SetActive(true);
				Demo.SetActive(false);
				OptionsGroup.SetActive(false);
				LevelComplete.SetActive(false);
				break;

			case States.Init:
				StartButton.SetActive(false);
				RestartButton.SetActive(false);
				QuitButton.SetActive(false);
				MainGroup.SetActive(false);
				BackPanel.SetActive(true);
				ScoreGroup.SetActive(false);
				LevelSelect.SetActive(false);
				BallSelect.SetActive(false);
				GetReadyGroup.SetActive(false);
				Blocks.SetActive(true);
				Logo.SetActive(true);
				WinGroup.SetActive(false);
				Demo.SetActive(false);
				OptionsGroup.SetActive(false);
				LevelComplete.SetActive(false);
				break;

			case States.Options:
				StartButton.SetActive(false);
				RestartButton.SetActive(false);
				QuitButton.SetActive(true);
				MainGroup.SetActive(false);
				BackPanel.SetActive(true);
				ScoreGroup.SetActive(false);
				LevelSelect.SetActive(false);
				BallSelect.SetActive(false);
				GetReadyGroup.SetActive(false);
				Blocks.SetActive(true);
				Logo.SetActive(false);
				WinGroup.SetActive(false);
				Demo.SetActive(false);
				OptionsGroup.SetActive(true);
				LevelComplete.SetActive(false);
				break;

			case States.LevelComplete:
				StartButton.SetActive(false);
				RestartButton.SetActive(false);
				QuitButton.SetActive(false);
				MainGroup.SetActive(false);
				BackPanel.SetActive(false);
				ScoreGroup.SetActive(false);
				LevelSelect.SetActive(false);
				BallSelect.SetActive(false);
				GetReadyGroup.SetActive(false);
				Blocks.SetActive(true);
				Logo.SetActive(false);
				WinGroup.SetActive(false);
				Demo.SetActive(false);
				OptionsGroup.SetActive(false);
				LevelComplete.SetActive(true);
				break;
		}

		ChildUpdate(typeof(GuiMain));
        ChildUpdate(typeof(GuiOSD));
		ChildUpdate(typeof(GuiLevelSelect));
	}

	public void StateChangeCallback(SceneLogic.StateChangeReason reason)
    {
        switch (reason)
        {
            case SceneLogic.StateChangeReason.Init:
				mState = States.Init;
				break;

            case SceneLogic.StateChangeReason.LevelSelect:
                mState = States.Main;
                break;

            case SceneLogic.StateChangeReason.Restart:
                mState = States.Main;
                break;

			case SceneLogic.StateChangeReason.Winner:
				mState = States.Win;
				break;

			case SceneLogic.StateChangeReason.LevelComplete:
				mState = States.LevelComplete;
				break;
        }

        UpdateState();
    }

    private void ChildUpdate(Type t)
    {
        var obj = FindObjectOfType(t) as IGuiUpdate;
        if (obj != null)
            obj.UpdateState();
    }

    private void Update()
    {
		switch (mState)
        {
		    case States.Init:
				mInitSeconds += Time.deltaTime;
				if (mInitSeconds > InitWaitTime)
				{
					// reset
					mInitSeconds = 0.0f;

					mState = States.Intro;
					UpdateState();
				}
				break;

			case States.Intro:
				mIntroWaitSeconds += Time.deltaTime;
				if (mIntroWaitSeconds > IntroWaitTime)
				{
					// reset
					mIntroWaitSeconds = 0.0f;

					mState = States.Demo;
					UpdateState();
				}
				break;

			case States.Demo:
				if (mSceneLogic != null && !mSceneLogic.Playing)
				{
					// start demo
					mSceneLogic.Play();
				}
				break;

			case States.GetReady:
				mGetReadySeconds += Time.deltaTime;
				if (mGetReadySeconds > GetReadyTime)
				{
					mGetReadySeconds = 0.0f;

					mState = States.OSD;
					UpdateState();

					if (mSceneLogic != null)
					{
						// start game
						mSceneLogic.Play();
					}
				}
				break;

			case States.Win:
				if (WinGroup != null)
				{
					var texts = WinGroup.GetComponentsInChildren<Text>();
					if (texts != null)
					{
						bool direction = false;
						foreach (var text in texts)
						{
							// colour cycle
							float h, s, v;
							Color.RGBToHSV(text.color, out h, out s, out v);
							h += Time.deltaTime * ColourSpeed;
							h %= 256;
							text.color = Color.HSVToRGB(h, s, v);

							// rotate
							var d = direction ? +1.0f : -1.0f;
							direction ^= true;
							text.transform.Rotate(0, 0, Time.deltaTime * RotateSpeed * d, Space.Self);
						}
					}
				}

				mWinSeconds += Time.deltaTime;
				if (mWinSeconds > WinTime)
				{
					// reset
					mWinSeconds = 0.0f;

					mState = States.Intro;
					UpdateState();
				}
				break;
		}
	}

    // private variables
    private enum States { Main, OSD, Restart, LevelSelect, BallSelect, GetReady, Intro, Demo, Win, Init, Options, LevelComplete };
    private States mState = States.Init;
    private float mInitSeconds, mGetReadySeconds, mWinSeconds, mIntroWaitSeconds, mCreditWaitSeconds;
	private SceneLogic mSceneLogic;
}
