﻿using UnityEngine;
using UnityEngine.UI;

public class SceneLogic : MonoBehaviour
{
	public int TargetFPS = 60;
	public int DebugMaxReachedLevel;
	public bool DebugSwitch;
	static public bool DebugMode
    {
        get
        {
			var me = GameObject.FindObjectOfType<SceneLogic>();
            return (Debug.isDebugBuild && !Application.isEditor) || (me.DebugSwitch && Application.isEditor);
        }
    }

	public enum CameraStyles { Fixed, Rotate };

	// public properties
	public GameObject MainCameraPole;
	public float CameraSpeed = 1.0f;
	public CameraStyles CameraStyle;
	public Level Level;
	public BallList BallList;
	public GameObject Shadow;
	public float Gravity = 4.0f;
    public float Speed = 1.0f;
    public float JumpFactor = 5.0f;
	public float ResetZ = 10.0f;
	public int StartNode = 0;
	public float TimeScale = 1.0f;
	public float PlayerClickSeconds = 0.1f; // 100 ms
	public bool Playing;
    public SfxLogic SfxLogic;
	public int FrontShow = 10;
	public int BackHide = 5;

	public void Play()
	{
		// play game
		Playing = true;
	}

	void Start()
	{
		Application.targetFrameRate = 60;

		StartLevel(StateChangeReason.Init);

		// request specific refresh?
		if (TargetFPS > 0.0f)
		{
			Application.targetFrameRate = TargetFPS;
		}

		// for debugging
		Time.timeScale = TimeScale;
	}

	void Update()
	{
		if (mBallZ > ResetZ)
		{
			// reset game
			StartLevel(StateChangeReason.Restart);
		}

		var speed = Playing ? Speed_Internal : 0.0f;
        var deltaTime = Playing ? Time.deltaTime : 0.0f;

        UpdateLogic(speed, deltaTime);

		if (Playing)
		{
			if (!mDemoMode)
				UpdateInput();
			else
				UpdateBot();
		}

		UpdateExtraLogic();
		UpdateLevelLogic();
	}

	public void LateUpdate()
	{
		if (!mLevelComplete)
		{
			UpdateCamera();
		}
	}

	private void UpdateLogic(float speed, float deltaTime)
	{
		var ball = mBall.GetComponentInChildren<Ball>();

		// obtain vectors etc. from level
		Vector3 from, to, ftv;
		float wdt, ldt;
		Level.UpdatePath(mPlayerNodeIdx, out from, out to, out ftv, out wdt, out ldt);

		if (mCalculateNextIndex)
		{
			// calculate next index
			mPlayerNodeNextIdx = Level.CalculateNextIndex(mPlayerNodeIdx);

			// change?
			if (mNodeIndexChange)
			{
                var blockType = Level.GetBlockAtIndex(mPlayerNodeIdx);

                if (blockType == Block.Blocks.Direction)
                {
                    if (mPlayerNodeIdx > 0)
                    {
                        // play direciton sound
                        SfxLogic.PlaySfx(SfxLogic.Sfx.Direction);
                    }
                }
                else if (blockType == Block.Blocks.Jump)
                {
                    // apply a jump velocity
                    PlayerJump(ldt, wdt, from, to);
				}

                // update level nodes
                Level.SetNextAction(mPlayerNodeNextIdx);
			}

			mCalculateNextIndex = false;
		}

		// set direction
		mDirection = ftv.normalized;

		// ballpos (1/3) before move
		var ballPos = new Vector3(mBall.transform.position.x, mBall.transform.position.y, mBall.transform.position.z + ball.VisualZOffset);   // remove visual offset

		// move ball in direction
		var v = mDirection * deltaTime * speed;
		ballPos += v;   // ball pos (2/3) after direction move
		mJumpedDistance += mJumping ? v.magnitude : 0.0f;

        // find current node for ball position (we ignore Z)
        mCurrentNode = Level.FindNode(ballPos);

		var block = mCurrentNode != null ? mCurrentNode.Block : null;
		var nodeValid = block != null ? block.InPlace : false;

		// move ball in Z
		var prevBallZ = mBallZ;
		mBallVelocityZ += Gravity * deltaTime;
		mBallZ += mBallVelocityZ * deltaTime;

		// ball pos (3/3) after Z move
		ballPos = new Vector3(ballPos.x, ballPos.y, mBallZ);

		// reset strobe
		mGroundContact = false;
			 
		if (!mLevelComplete)
		{
			// ball [potentially] moving through ground?
			if (mBallZ > 0.0f && prevBallZ <= 0.0f)
			{
				if (mJumping && mJumpLandingAt != Vector2.zero)
				{
					// where ball should be on ground (to correct jump inaccuracies)
					var tmpBallPos = new Vector3(mJumpLandingAt.x, mJumpLandingAt.y, mBallZ);

					// reset
					mJumpLandingAt = Vector2.zero;

					// find current node for ball position (we ignore Z)
					mCurrentNode = Level.FindNode(tmpBallPos);

					block = mCurrentNode != null ? mCurrentNode.Block : null;
					nodeValid = block != null ? block.InPlace : false;

					if (mCurrentNode != null && nodeValid)
					{
						// assign to actual ball
						ballPos = tmpBallPos;
					}
				}

				// check if we hit ground (node has to be valid to be "hit")
				if (mCurrentNode != null && nodeValid)
				{
					if (mJumping)
					{
                        // play "hit ground" sound
                        SfxLogic.PlaySfx(SfxLogic.Sfx.Thud);
						mGroundContact = true;
					}

					// hit ground
					mBallZ = 0.0f;
					mBallVelocityZ = 0.0f;

					mJumping = false;
					mJumpDistance = 0.0f;
					mJumpedDistance = 0.0f;
				}
				else if (!mLevelComplete)
				{
					// missed ground - FAIL!

					// play fail sound
					SfxLogic.PlaySfx(SfxLogic.Sfx.Fail);
				}
			}
		}

		var isBallOnGround = mBallZ == 0.0f;

		// place ball
		mBall.transform.position = ballPos + new Vector3(0, 0, mBallZ - ball.VisualZOffset);	// apply visual offset

		// place shadow
		mShadow.transform.position = new Vector3(ballPos.x, ballPos.y, kShadowZ);
		mShadow.SetActive(mBallZ <= 0.0f);

		// assign back to ball properties
		ball.Velocity = v;
		ball.Direction = mDirection;
		ball.OnGround = isBallOnGround;
	}

	private void UpdateExtraLogic()
	{
		if (mCurrentNode != null)
		{
			var node = mCurrentNode;
			var block = node.Block;
			var blockType = block.ChosenBlock;
			var ball = mBall.GetComponentInChildren<Ball>();

			// finished?
			if (blockType == Block.Blocks.Finish)
			{
				if (!mLevelComplete && ball.OnGround)
				{
					// play winner sound
					SfxLogic.PlaySfx(SfxLogic.Sfx.LevelComplete);

					// fire ball into space
					mBallVelocityZ = -kLevelUpZVelocity * JumpFactor_Internal;
					mJumping = true;

					mLevelComplete = true;

					// execute callback to UI
					FindObjectOfType<GuiLogic>().StateChangeCallback(StateChangeReason.LevelComplete);
				}
			}
			else
			{
				if (mPlayerNodeNextIdx == node.Index && ball.OnGround && blockType != Block.Blocks.Blank)
				{
					// set highlight
					Level.SetHighlight(node.Index);
				}
				else
				{
					// no highlight
					Level.SetHighlight(-1);
				}
			}
		}
		else
		{
			// no highlight
			Level.SetHighlight(-1);
		}
	}

	private void UpdateLevelLogic(bool instant = false)
	{
		var node = mCurrentNode;
		var block = node != null ? node.Block : null;
		var ball = mBall.GetComponentInChildren<Ball>();

		// node has to be valid (or instant mode) to enable logic
		var nodeValid = instant || ((block != null && ball != null) ? (block.InPlace && ball.OnGround) : false);

		if (node != null && nodeValid)
		{
			// enable - start to show blocks in front of player
			bool ok = true;
			var index = node.Index;
			var ni = 0;
			while (ok)
			{
				var nextNode = Level.GetNodeAtIndex(index + ni++);
				if (nextNode != null && ni < FrontShow)
				{
					nextNode.Block.Show(instant);
					nextNode.Block.gameObject.SetActive(true);	// enable Update()
				}
				else
				{
					ok = false;
				}
			}

			// disable - remove blocks behind player
			ok = true;
			index = node.Index - BackHide;
			ni = 0;
			while (ok)
			{
				var prevNode = Level.GetNodeAtIndex(index - ni++);
				if (prevNode != null)
				{
					prevNode.Block.Hide(instant);
				}
				else
				{
					ok = false;
				}
			}
		}
	}

	private void UpdateCamera()
	{
		var ballPos = mBall.transform.position;

		// align cameras with ball
		var x = ballPos.x;
		var y = ballPos.y;
		var z = Mathf.Min(0, ballPos.z);

		if (MainCameraPole != null)
		{
			// position camera pole base on player
			MainCameraPole.transform.position = new Vector3(x, y, z);

			// rotation camera around player
			if (mSnapCamera)
			{
				// instant
				MainCameraPole.transform.rotation = mBall.transform.rotation;
				mSnapCamera = false;
			}
			else if (CameraStyle == CameraStyles.Rotate)
			{
				// smooth
				MainCameraPole.transform.rotation = Quaternion.Lerp(MainCameraPole.transform.rotation, mBall.transform.rotation, Time.deltaTime * CameraSpeed);
			}
		}
	}

	private void UpdateBot()
	{
		var ball = mBall.GetComponentInChildren<Ball>();
		var isBallOnGround = ball.OnGround;
		var playerPress = Input.GetMouseButtonDown(kLeftButton);

		if (isBallOnGround)
		{
			// on action block?
			if (mCurrentNode != null && mCurrentNode.Index == mPlayerNodeNextIdx)
			{
				var targetPt = Level.GetNodeAtIndex(mPlayerNodeNextIdx).Obj.transform.position;
				var ballPos = mBall.transform.position;
				var ballPt = new Vector3(ballPos.x, ballPos.y, 0);
				var delta = (targetPt - ballPt).magnitude;

				// within tollerance or coming off a jump?
				if (delta < 0.5f || mGroundContact)
				{
					PlayerMove();
				}
			}
		}
	}

	private void UpdateInput()
	{
		var ball = mBall.GetComponentInChildren<Ball>();
		var isBallOnGround = ball.OnGround;
		var playerPress = Input.GetMouseButtonDown(kLeftButton);

		// check user input
		if (playerPress)
		{
			if (isBallOnGround)
			{
                // on action block?
                if (mCurrentNode != null && mCurrentNode.Index == mPlayerNodeNextIdx)
                {
                    // player action
					PlayerMove();
				}
				else
                {
                    // player jump
                    float ljmp = kLogicalSmallJump;
                    PlayerJump(ljmp, Level.LogicalToWorld(ljmp), Vector2.zero, Vector2.zero);
                }
			}
			else if (mLastPlay == 0.0f)
			{
				// pend action
				mLastPlay = Time.time;
			}
		}
		else if (mLastPlay > 0.0f && isBallOnGround)
		{
			if (Mathf.Abs(Time.time - mLastPlay) < PlayerClickSeconds)
			{
				// within allowance, action now for player
				PlayerMove();
			}

			// reset
			mLastPlay = 0.0f;
		}
	}

	public enum StateChangeReason { Init, LevelSelect, Restart, Winner, LevelComplete };
	public void StartLevel(StateChangeReason reason)
	{
		if (reason == StateChangeReason.LevelSelect)
		{
			// stop any demo
			mDemoMode = false;
		}

		if (!mDemoMode)
		{
			if (Application.isEditor && DebugMaxReachedLevel > 0)
			{
				// apply and reset
				Level.CurrentLevel = Level.UnlockedLevel = DebugMaxReachedLevel;
				DebugMaxReachedLevel = 0;
			}

			// next level?
			if (mLevelComplete)
			{
				Level.CurrentLevel++;
				if (!Level.DoesLevelExist(Level.CurrentLevel))
				{
					// we assume that a missing level means the game is won (i.e. all levels complete)
					reason = StateChangeReason.Winner;

					// reset back to the first level
					Level.CurrentLevel = 1;
				}
			}

			Level.UnlockedLevel = Mathf.Max(Level.CurrentLevel, Level.UnlockedLevel);

			// (re)build level
			Level.BuildLevel();
		}
		else
		{
			// (re)build level
			Level.BuildLevel(Level.DemoLevel);

			// switch back to intro
			reason = StateChangeReason.Init;
		}

		// reset our logic
		Reset();

		// execute callback to UI
        FindObjectOfType<GuiLogic>().StateChangeCallback(reason);
	}

	private void Reset()
	{
		// reset
		mPlayerNodeIdx = StartNode;
		mPlayerNodeNextIdx = 0;
		mDirection = Vector3.zero;
		mCalculateNextIndex = mNodeIndexChange = true;
		mBallZ = 0.0f;
		mBallVelocityZ = 0.0f;
        mJumping = false;
        mJumpDistance = mJumpedDistance = 0.0f;
		mJumpLandingAt = Vector2.zero;
		mCurrentNode = null;
		mLevelComplete = false;
		mSnapCamera = true;
		mGroundContact = false;

		// re-create the ball + shadow
		InstantiateBall();

		// update the level logic
		UpdateLevelLogic(true);

		// halt until play started
		Playing = false;
	}

	public void InstantiateBall()
	{
		// obtain vectors etc. from level
		Vector3 from, to, ftv;
		float wdt, ldt;
		Level.UpdatePath(mPlayerNodeIdx, out from, out to, out ftv, out wdt, out ldt);

		if (mBall != null)
			GameObject.Destroy(mBall);
		if (mShadow != null)
			GameObject.Destroy(mShadow);

		// create ball from prefab
		mBall = (GameObject)GameObject.Instantiate(BallList.SelectedBall, transform);
		mShadow = (GameObject)GameObject.Instantiate(Shadow, transform);

		// set initial ball position
		mBall.transform.position = from;

		var ball = mBall.GetComponentInChildren<Ball>();

		// set initial ball direction
		mDirection = ftv.normalized;
		ball.Direction = mDirection;

		// find current node for ball position (we ignore Z)
		mCurrentNode = Level.FindNode(mBall.transform.position, true);
	}

	private void PlayerJump(float ldt, float wdt, Vector3 from, Vector3 to)
    {
        // initiate a jump
        mJumping = ldt > 1.0f;
		mJumpDistance = mJumping ? wdt : 0.0f;
        mJumpedDistance = 0.0f;
        mBallVelocityZ = mJumping ? (-ldt * JumpFactor_Internal) : 0.0f;

		if (mJumping)
		{
			// play jump sound
			SfxLogic.PlaySfx(SfxLogic.Sfx.Bounce);

			if (from != Vector3.zero && to != Vector3.zero)
			{
				// work out where the ball should land
				var offset = from - mBall.transform.position;
				mJumpLandingAt = to - offset;
			}
			else
			{
				mJumpLandingAt = Vector2.zero;
			}
		}
		else
		{
			mJumpLandingAt = Vector2.zero;
		}
	}

    private void PlayerMove()
	{
        var nextBlockType = Level.GetBlockAtIndex(mPlayerNodeNextIdx);
        if (nextBlockType != Block.Blocks.Finish)
        {
            // apply next move
            mNodeIndexChange = mPlayerNodeIdx != mPlayerNodeNextIdx;
            mPlayerNodeIdx = mPlayerNodeNextIdx;

            // trigger re-calculate on next frame
            mCalculateNextIndex = true;
        }
    }

    private float Speed_Internal
    {
        get { return Speed * Level.DifficultyFactor; }
    }

    private float JumpFactor_Internal
    {
        get { return JumpFactor / Level.DifficultyFactor; }
    }

	// private constants
	private const float kShadowZ = 0.0f;
	private const int kLeftButton = 0;  // maps to left mouse button (or single touch on mobile)
	private const float kLevelUpZVelocity = 4.0f;
    private const float kLogicalSmallJump = 3.0f;

    // private variables
    private int mPlayerNodeIdx, mPlayerNodeNextIdx;
	private GameObject mBall, mShadow;
	private Vector3 mDirection;
	private bool mCalculateNextIndex, mNodeIndexChange;
	private float mBallZ, mBallVelocityZ;
	private bool mJumping;
	private float mJumpDistance, mJumpedDistance;
	private Vector2 mJumpLandingAt;
	private float mLastPlay = 0.0f;
	private Node mCurrentNode;
	private bool mLevelComplete;
	private bool mSnapCamera;
	private bool mGroundContact;
	private bool mDemoMode = true;
}
