﻿using UnityEngine;
using System;
using System.Linq;
using System.Collections.Generic;

public class Level : MonoBehaviour
{
	public const int LevelsPerDifficulty = 5;
	public const int MaxLevels = 20;

	public int[] DemoLevels;

	// public properties
	public float DifficultyFactor = 1.0f;

    public GameObject Block;
	public GameObject Blocks;
	public VertexColours VertexColours;

	public string[] RankDescriptions = new string[] { "None" };

	public string GetRankForLevel(int level)
	{
		int rankIndex = (level - 1) / Level.LevelsPerDifficulty;
		return RankDescriptions[rankIndex % RankDescriptions.Length];
	}

	public string GetRankString(int rankIndex)
	{
		return RankDescriptions[rankIndex % RankDescriptions.Length];
	}

	public static int CurrentLevel
	{
		get { return PlayerPrefs.GetInt("CurrentLevel", 1); }
		set { PlayerPrefs.SetInt("CurrentLevel", value); }
	}

	public static int UnlockedLevel
	{
		get { return PlayerPrefs.GetInt("UnlockedLevel", 1); }
		set { PlayerPrefs.SetInt("UnlockedLevel", value); }
	}

	private void Reset()
	{
		// reset all blocks
		foreach (var node in mNodes)
		{
			var block = node.Block;
			block.Setup();
			block.SetNextAction(false);
			block.Hide(true);
			block.gameObject.SetActive(false);
		}

        mLastActionIndex = mLastHighlightIndex = -1;
    }

	public int CalculateNextIndex(int index)
	{
		var oldIndex = index;

		// find next block index of play
		bool exit = false;
		do
		{
			index++;

            // sanity
            if (index >= mNodes.Count)
            {
                return oldIndex;
            }
            
            // get block and check
			var block = mNodes[index].Block;
			exit = block.ChosenBlock == global::Block.Blocks.Direction 
                || block.ChosenBlock == global::Block.Blocks.Jump
                || block.ChosenBlock == global::Block.Blocks.Finish;
		}
		while (!exit);

		return index;
	}

    public global::Block.Blocks GetBlockAtIndex(int index)
    {
        if (index < 0 || index >= mNodes.Count)
        {
            return global::Block.Blocks.Blank;
        }

        return mNodes[index].Block.ChosenBlock;
    }

	public Node GetNodeAtIndex(int index)
	{
		if (index < 0 || index >= mNodes.Count)
		{
			return null;
		}

		return mNodes[index];
	}

	public void SetNextAction(int index)
	{
		// change?
		if (mLastActionIndex != index)
		{
            // remove
            if (mLastActionIndex != -1)
            {
                mNodes[mLastActionIndex].Block.SetNextAction(false);
            }

            // apply
            if (index != -1)
            {
                mNodes[index].Block.SetNextAction(true);
            }

            mLastActionIndex = index;
        }
    }

    public void SetHighlight(int index)
    {
        // change?
        if (mLastHighlightIndex != index)
        {
            // remove
            if (mLastHighlightIndex != -1)
            {
                mNodes[mLastHighlightIndex].Block.SetHighlight(false);
			}

            // apply
            if (index != -1)
            {
                mNodes[index].Block.SetHighlight(true);
            }

            mLastHighlightIndex = index;
        }
    }

	public void UpdatePath(int index, out Vector3 from, out Vector3 to, out Vector3 ftv, out float wdt, out float ldt)
	{
		// obtain play information from index

		var maxValidIndex = mNodes.Count - 1;
		var n0 = Mathf.Min(index, maxValidIndex);
		var n1 = Mathf.Min(index + 1, maxValidIndex);

		from = mNodes[n0].Obj.transform.position;
		to = mNodes[n1].Obj.transform.position;

		ftv = (to - from);

		wdt = mNodes[n1].WorldDistance;
		ldt = mNodes[n1].LogicalDistance;
	}

	public Node FindNode(Vector3 worldPosition, bool nonColliderMethod = false)
	{
		Node rv = null;

		if (nonColliderMethod)
		{
			// XXX: WARNING - LINEAR SEARCH - VERY INEFFICIENT!!

			// find node that is under player
			foreach (var node in mNodes)
			{
				var obj = node.Obj;
				var objT = obj.transform;

				var itp = objT.InverseTransformPoint(new Vector3(worldPosition.x, worldPosition.y, kBlockZ));
				if (Math.Abs(itp.x) < kSpacingSize / 2 && Math.Abs(itp.y) < kSpacingSize / 2)
				{
					// found
					rv = node;
					break;
				}
			}
		}
		else
		{
			// use colliders to find node via object (hoping more efficient that linear search above, assuming a collider on the block root gameobject)
			RaycastHit hitInfo;
			if (Physics.Raycast(new Vector3(worldPosition.x, worldPosition.y, -1.0f), Vector3.forward, out hitInfo))
			{
				var blockObj = hitInfo.transform.gameObject;
				var block = blockObj.GetComponent<Block>();
				return block.Node;
			}
		}

		return rv;
	}

    public float LogicalToWorld(float ldt)
    {
        return ldt * kSpacingSize;
    }

    public float WorldToLogical(float wdt)
    {
        return wdt / kSpacingSize;
    }

	private void Clear(GameObject obj)
	{
		// clear any blocks
		for (int c = 0; c < obj.transform.childCount; ++c)
		{
			GameObject.Destroy(obj.transform.GetChild(c).gameObject);
		}
	}

	private string GetLevelResource(int levelNumber)
	{
		return string.Format("Levels/Level {0}", levelNumber);
	}

	public bool DoesLevelExist(int levelNumber)
	{
		// do we have data for this level?
		var levelData = Resources.Load(GetLevelResource(levelNumber)) as TextAsset;
		return levelData != null;
	}

	public int DemoLevel
	{
		get
		{
			if (mDemoLevel == -1)
			{
				// choose demo starting level
				mDemoLevel = UnityEngine.Random.Range(0, DemoLevels.Length);
			}

			var rv = mDemoLevel;
			if (++mDemoLevel >= DemoLevels.Length)
			{
				mDemoLevel = 0;
			}

			return DemoLevels[mDemoLevel];
		}
	}

	public void BuildLevel(int level = -1)
	{
		var root = Blocks;

		Clear(root);

		// load level data
		level = level == -1 ? CurrentLevel : level;
		var levelData = Resources.Load(GetLevelResource(level)) as TextAsset;
		if (levelData == null)
		{
			return;
		}

		// update colours
		if (VertexColours != null)
		{
			VertexColours.ColourIndex = (level-1) / LevelsPerDifficulty;
		}

		List<string> lines = TextAssetToList(levelData);
		mNodes = CreateNodeList(lines).ToList();

		// traverse nodes and calculate block attributes
		Node lastNode = null;
		Vector2 lastDirection = Vector2.up;
		float lastAngle = 0.0f;
		bool lastIsJump = false;
		global::Block.Backgrounds currentBackground = global::Block.Backgrounds.UBend;

		// XXX: WARNING - HACKS-R-US ALL OVER THE PLACE!

		// setup node list
		int nodeCount = 1;
		foreach (var node in mNodes)
		{
			// create new object:
			node.Obj = (GameObject)GameObject.Instantiate(Block, root.transform);
			node.Obj.name = "Block_" + nodeCount;
			node.Obj.transform.localPosition = new Vector3(node.LogicalX * kSpacingSize, node.LogicalY * kSpacingSize, kBlockZ);
			node.Block = node.Obj.GetComponentInChildren<Block>();
			node.Block.Node = node;

			if (lastNode != null)
			{
				var direction = new Vector2(node.LogicalX, node.LogicalY) - new Vector2(lastNode.LogicalX, lastNode.LogicalY);
				var angle = Mathf.Atan2(direction.y, direction.x) * Mathf.Rad2Deg;
				var lastBlock = lastNode.Block;

				bool isJump = direction.magnitude > 1.0f;
				bool isFinish = nodeCount == mNodes.Count;

				var cross = Vector3.Cross(lastDirection, direction);
				var backgroundAngle = cross.z > 0.0f ? angle + 90 : angle;

				if (nodeCount > 2 && (angle != lastAngle || isJump || lastIsJump))
				{
					lastBlock.Rotation = angle;
					lastBlock.ChosenBlock = isJump ? global::Block.Blocks.Jump : global::Block.Blocks.Direction;
					if (!isJump && lastIsJump && angle == lastAngle)
					{
						lastBlock.ChosenBlock = global::Block.Blocks.Blank;
					}

					lastBlock.ChosenBackground = lastIsJump || isJump ? global::Block.Backgrounds.UBend : global::Block.Backgrounds.Corner;

					if (lastIsJump && isJump)
					{
						lastBlock.ChosenBackground = global::Block.Backgrounds.Circle;
					}

					lastBlock.BackgroundRotation = isJump ? 270 + lastAngle : (lastIsJump ? 90 + angle : backgroundAngle);
				}
				else
				{
					lastBlock.ChosenBlock = global::Block.Blocks.Blank;
					lastBlock.ChosenBackground = currentBackground;
					currentBackground = global::Block.Backgrounds.Default;
					lastBlock.BackgroundRotation = 90 + angle;
				}

				node.WorldDistance = direction.magnitude * kSpacingSize;
				node.LogicalDistance = direction.magnitude;

				if (isFinish)
				{
					var block = node.Block;

					block.ChosenBlock = global::Block.Blocks.Finish;
					block.ChosenBackground = global::Block.Backgrounds.UBend;
					block.BackgroundRotation = 270 + angle;
					block.Rotation = lastAngle - 90;

					if (isJump)
					{
						block.ChosenBackground = global::Block.Backgrounds.Circle;
					}
				}

				lastDirection = direction;
				lastAngle = angle;
				lastIsJump = isJump;
			}

			nodeCount++;
			lastNode = node;
		}

		Reset();
	}

	private IEnumerable<Node> CreateNodeList(List<string> lines)
	{
		var nodes = new List<Node>();
		int y = lines.Count - 1;
		int index = 0;

		// for each line (y)
		foreach (var line in lines)
		{
			// for each col (x)
			string[] components = line.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
			int x = 0;
			foreach (var component in components)
			{
				if (int.TryParse(component, out index) && index > 0)
				{
					// create node
					var node = new Node() { Index = (index-1), LogicalX = x, LogicalY = y };    // the -1 converts from (1...x) data to (0...x) array index

					// add to node list
					nodes.Add(node);
				}

				x++;
			}

			y--;
		}

		// sort by travel order
		return nodes.OrderBy(n => n.Index);
	}

	private List<string> TextAssetToList(TextAsset ta)
	{
		// read text and convert into a List<string>
		var listToReturn = new List<string>();
		var arrayString = ta.text.Split('\n');
		foreach (var line in arrayString)
		{
			listToReturn.Add(line);
		}

		return listToReturn;
	}

	// private constants
	private const float kBlockZ = 0.0f;
	private const float kSpacingSize = 2.5f;

	// private variables
	private List<Node> mNodes = new List<Node>();
    private int mLastActionIndex = -1;
    private int mLastHighlightIndex = -1;
	private int mDemoLevel = -1;
}

public class Node
{
	public int Index { get; set; }
	public int LogicalX { get; set; }
	public int LogicalY { get; set; }
	public GameObject Obj { get; set; }
	public Block Block { get; set; }
	public float WorldDistance { get; set; }
	public float LogicalDistance { get; set; }
}
