﻿using UnityEngine;

public class Block : MonoBehaviour
{
    // public properties
    public GameObject Gfx;
    public GameObject Background;

	public Sprite[] GfxList;
    public Color[] GfxColours;
    public Sprite[] BackgroundList;

    public enum Blocks { Blank, Direction, Jump, Finish };
	public Blocks ChosenBlock;

    public enum Backgrounds { Default, Corner, UBend, Circle };
    public Backgrounds ChosenBackground;

	public enum ShowModes { Always, Dynamic };
	public ShowModes ShowMode;

	public float Rotation;
    public float BackgroundRotation;

	public float InactiveActionAlpha = 0.25f;
	public float ActiveActionAlpha = 1.0f;

    public float NormalScale = 1.0f;
    public float HighlightScale = 1.5f;

	public float ShowSpeed = 1.0f;
	public float ZFloor = 100.0f;

	public Node Node
	{
		get; set;
	}

	public void Setup()
    {
        if (Gfx != null)
        {
			// setup gfx
            var s = Gfx.GetComponent<SpriteRenderer>();
            s.sprite = GfxList[Mathf.Min(GfxList.Length - 1, (int)ChosenBlock)];
            mColor = GfxColours[Mathf.Min(GfxColours.Length - 1, (int)ChosenBlock)];
			s.color = mColor;
			Gfx.transform.rotation = Quaternion.Euler(0, 0, Rotation);
        }

        if (Background != null)
        {
			// setup background
            var s = Background.GetComponent<SpriteRenderer>();
            s.sprite = BackgroundList[Mathf.Min(BackgroundList.Length - 1, (int)ChosenBackground)];
            Background.transform.rotation = Quaternion.Euler(0, 0, BackgroundRotation);
        }

        mNextAction = mHighlight = false;
    }

	public void SetNextAction(bool aSet)
	{
        mNextAction = aSet;
		InternalUpdate();
    }

    public void SetHighlight(bool aSet)
    {
        mHighlight = aSet;
		InternalUpdate();
    }

	public void Show(bool instant = false)
	{
		mShown = true;
		if (instant)
		{
			mT = 1.0f;
			InternalUpdate();
		}
	}

	public void Hide(bool instant = false)
	{
		mShown = false;
		if (instant)
		{
			mT = 0.0f;
			InternalUpdate();
		}
	}

	public bool InPlace
	{
		get
		{
			var brt = BlockRootTransform;
			return brt.position.z == 0.0f;
		}
	}

	void Update()
	{
		if (mShown && mT < 1.0f)
		{
			// move block toward placement
			mT += Time.deltaTime * ShowSpeed;
			if (mT > 1.0f)
			{
				mT = 1.0f;
			}

			InternalUpdate();
		}
		else if (!mShown && mT > 0.0f)
		{
			// move block away from placement
			mT -= Time.deltaTime * ShowSpeed;
			if (mT < 0.0f)
			{
				mT = 0.0f;
			}

			InternalUpdate();
		}
	}

	private void InternalUpdate()
	{
		UpdateBackground();
		UpdateGfx();

		if (mT == 0.0f)
		{
			gameObject.SetActive(false);	// disable Update()
		}
	}

	private void UpdateBackground()
	{
		// colour + alpha
		var s = Background.GetComponent<SpriteRenderer>();
		var c = s.color;
		s.color = new Color(c.r, c.g, c.b, ShowMode == ShowModes.Dynamic ? mT : 1.0f);

		// position
		var z = Mathf.Lerp(ZFloor, 0, mT);
		var brt = BlockRootTransform;

		if (ShowMode == ShowModes.Dynamic)
		{
			brt.position = new Vector3(brt.position.x, brt.position.y, z);
			brt.gameObject.SetActive(z != ZFloor);
		}
		else
		{
			brt.gameObject.SetActive(true);
		}
	}

	private void UpdateGfx()
    {
		// colour + alpha
        var s = Gfx.GetComponent<SpriteRenderer>();
        var c = mColor;
        c.a = (mNextAction || mHighlight) ? ActiveActionAlpha : InactiveActionAlpha;
		c.a *= mT;
        s.color = c;

		// scale
        float sc = mHighlight ? HighlightScale : NormalScale;
        s.transform.localScale = new Vector3(sc, sc, sc);
    }

	private Transform BlockRootTransform
	{
		get { return transform.GetChild(0).transform; }	// XXX: fragile!
	}

    // private variables
    private bool mNextAction;
    private bool mHighlight;
	private bool mShown;
	private float mT;
	private Color mColor;
}
