﻿using System.Collections.Generic;
using System.IO;
using System.Linq;
using UnityEditor;
using UnityEngine;

public class LvlEditor : EditorWindow
{
	private static LvlEditor mInstance = null;

	const string KSavePath = "Assets/Resources/Levels/";

	static string LevelName;

	float TileSize = 24;
	static int LevelWidth = 32;
	static int LevelHeight = 32;
	Vector2 MapScrollPos;

	static Texture2D mTileTex;
	static Texture2D mBackTex;

	static Rect rectFinish = new Rect(0,0,0.5f,0.5f);
	static Rect rectJump = new Rect(0.5f,0.5f,0.5f,0.5f);
	static Rect rectDir = new Rect(0,0.5f,0.5f,0.5f);
	static Rect rectBlank = new Rect(0.5f,0,0.5f,0.5f);

	static Rect rectBackDir = new Rect(0,0,0.5f,0.5f);

	static List<string> mLevels;

	bool mDirty;
	static string mSeed;

	List<EditorBlock> mSelection = new List<EditorBlock>();

	[MenuItem("Tools/Level Editor")]
	private static void Init()
	{
		mInstance = GetWindow<LvlEditor>("Level Editor");
		LevelName = "NewLevel";

		mTileTex = AssetDatabase.LoadAssetAtPath<Texture2D>("assets/sprites/actions.png");
		mBackTex = AssetDatabase.LoadAssetAtPath<Texture2D>("assets/sprites/tiles.png");

		mLevels = Directory.GetFiles(KSavePath, "*.txt").OrderBy(l => l.Length).ThenBy(l => l).ToList();

		RandomiseSeed();
	}

	private void OnGUI()
	{
		float top = 48;
		float div = position.width / 3 * 2;

		GUILayout.BeginHorizontal();
		{
			GUILayout.BeginVertical(GUILayout.Width(div));
				DrawEditorControls();
				DrawMap(top, div);
			GUILayout.EndVertical();

			GUILayout.BeginVertical();
				EditorGUILayout.LabelField("LMB = add block");
				EditorGUILayout.LabelField("RMB = delete block");
				EditorGUILayout.LabelField("MMB = de/select island");
				EditorGUILayout.LabelField("Cursors = move selection");
		
				// Level buttons:
				GUILayout.BeginVertical();
					for (int l=0 ; l<mLevels.Count ; l+=5)
					{
						GUILayout.BeginHorizontal();
							for (int il=l ; il<l+5 ; il++)
							{
								string path = mLevels[il];
								if (GUILayout.Button(Path.GetFileNameWithoutExtension(path)))
									LoadLevel(path);
							}
						GUILayout.EndHorizontal();
					}
				GUILayout.EndVertical();

				// Generator:
				GUILayout.BeginVertical();
					mSeed = EditorGUILayout.TextField("Seed", mSeed, GUILayout.MinWidth(300));
					mDifficulty = EditorGUILayout.Slider("Difficulty", mDifficulty, 0, 1);
					mLength = EditorGUILayout.Slider("Length", mLength, 0, 1);
					if (GUILayout.Button("Generate from seed"))
					{
						Clear();
						mBlocks = new LevelGen().Generate(mSeed, mDifficulty, mLength, LevelWidth, LevelHeight);
						LevelName = mSeed;
						OnBlocksChanged();
					}
					if (GUILayout.Button("New Random"))
					{
						RandomiseSeed();
						Clear();
						mBlocks = new LevelGen().Generate(mSeed, mDifficulty, mLength, LevelWidth, LevelHeight);
						LevelName = mSeed;
						OnBlocksChanged();
					}
				GUILayout.EndVertical();

			GUILayout.EndVertical();
		}
	}

	float mDifficulty;
	float mLength;

	private void DrawEditorControls()
	{
		GUILayout.BeginHorizontal();
			LevelName = EditorGUILayout.TextField("Name", LevelName, GUILayout.MinWidth(300));
			EditorGUILayout.LabelField(mDirty ? "unsaved" : "");
		GUILayout.EndHorizontal();

		GUILayout.BeginHorizontal();
			if (GUILayout.Button("Load"))
			{
				Load();
			}
			if (GUILayout.Button("Save"))
			{
				Save(LevelName);
				mLevels = Directory.GetFiles(KSavePath, "*.txt").OrderBy(l => l.Length).ThenBy(l => l).ToList();
			}
			if (GUILayout.Button("Clear"))
			{
				Clear();
			}
		GUILayout.EndHorizontal();
	}

	private static void RandomiseSeed()
	{
		Random.InitState((int)System.DateTime.Now.Ticks);
		mSeed = ((int)(Random.value * int.MaxValue)).ToString();
	}

	private void Clear()
	{
		bool clear = true;
		if (mDirty)
			clear = EditorUtility.DisplayDialog("Clear", "Unsaved work!", "Lose it", "Keep it");
		if (clear)
		{
			mBlocks.Clear();
			OnBlocksChanged();
			mDirty = false;
		}
	}

	private void Load()
	{
		if (mDirty)
		{
			if (!EditorUtility.DisplayDialog("Load level", "Discard changes?", "Discard", "Keep"))
				return;
		}

		string file = EditorUtility.OpenFilePanel("Load level", KSavePath, "txt");
		LoadLevel(file);
	}

	private void LoadLevel(string file)
	{
		if (!string.IsNullOrEmpty(file))
		{
			LoadData(file);
			OnBlocksChanged();
			mDirty = false;
			LevelName = Path.GetFileNameWithoutExtension(file);
		}
	}

	private void Save(string aLevelName)
	{
		string name = aLevelName + ".txt";
		var fullPath = Path.Combine(KSavePath, name);
		if (File.Exists(fullPath))
		{
			if (!EditorUtility.DisplayDialog("Save file?", "Overwrite " + aLevelName, "Overwrite", "No!"))
				return;
		}

		SaveData(fullPath);
		mDirty = false;
		AssetDatabase.Refresh();
	}

	private void SaveData(string aPath)
	{
		var lines = new List<string>(LevelHeight);
		for (int y=0 ; y<LevelHeight ; y++)
		{
			string line = "";
			for (int x=0 ; x<LevelWidth ; x++)
			{
				var block = mBlocks.FirstOrDefault(b => b.x==x && b.y==y);
				line += (block == null ? 0 : block.index+1).ToString("D3") + ",";
			}
			lines.Add(line);
		}
		File.WriteAllLines(aPath, lines.ToArray());
	}

	private void LoadData(string aPath)
	{
		var blocks = new List<EditorBlock>(100);
		var lines = File.ReadAllLines(aPath);
		for (int y=0 ; y<lines.Length ; y++)
		{
			var split = lines[y].Split(',');
			for (int x=0 ; x<split.Length ; x++)
			{
				int i;
				if (int.TryParse(split[x], out i) && i>0)
				{
					blocks.Add(new EditorBlock(i-1, x, y));
				}
			}
		}
		
		mBlocks = blocks.OrderBy(b => b.index).ToList();
	}

	bool mScrolled;
	private void DrawMap(float top, float div)
	{
		var mapRect = new Rect(8, top, div, position.height - top);

		// Input
		Event e = Event.current;
		bool isOverMap = mapRect.Contains(e.mousePosition);
		if (e.isMouse)
		{
			if (e.type == EventType.MouseDown && GUIUtility.keyboardControl != 0)
			{
				GUIUtility.keyboardControl = 0;	// Removes focus from gui controls (i.e. the name editor!)
				return;
			}

			int x = (int)((e.mousePosition.x - mapRect.xMin + MapScrollPos.x) / TileSize);
			int y = (int)((e.mousePosition.y - mapRect.yMin + MapScrollPos.y) / TileSize);

			if (e.button == 0)
			{
				if (e.type == EventType.MouseDrag)
				{
					MapScrollPos -= e.delta;
					mScrolled = true;
					Repaint();
				}
				else
				{
					if (e.type == EventType.MouseUp && isOverMap && !mScrolled)
					{
						if (x>=0 && x<LevelWidth && y>=0 && y<LevelHeight)
						{
							CheckAddTile(x,y);
						}
					}
					mScrolled = false;
				}
			}
			else if (e.button == 1 && e.type == EventType.MouseDown && isOverMap)
			{
				if (x>=0 && x<LevelWidth && y>=0 && y<LevelHeight)
				{
					var block = mBlocks.FirstOrDefault(b => b.x==x && b.y==y);
					if (block != null)
						RemoveBlock(block.index);
				}
			}
			else if (e.button == 2  && e.type == EventType.MouseDown && isOverMap)
			{
				SelectIsland(x, y);
			}
		}
		else if (e.type == EventType.KeyDown)
		{
			if (e.keyCode == KeyCode.Equals)
				Zoom(true);
			else if (e.keyCode == KeyCode.Minus)
				Zoom(false);
			else if (e.keyCode == KeyCode.LeftArrow)
				MoveSelection(-1,0);
			else if (e.keyCode == KeyCode.DownArrow)
				MoveSelection(0,1);
			else if (e.keyCode == KeyCode.UpArrow)
				MoveSelection(0,-1);
			else if (e.keyCode == KeyCode.RightArrow)
				MoveSelection(1,0);
			else if (e.keyCode == KeyCode.Escape)
			{
				mSelection.Clear();
				Repaint();
			}
		}
		else if (e.type == EventType.ScrollWheel && isOverMap)
		{
			Zoom(e.delta.y < 0);
		}

		int pixWidth = (int)(LevelWidth*TileSize);
		int pixHeight = (int)(LevelHeight*TileSize);

		MapScrollPos = GUI.BeginScrollView(mapRect,
											MapScrollPos,
											new Rect(0, 0, Mathf.Max(pixWidth+8, div-16), Mathf.Max(pixHeight+8, position.height-top)),
											true,
											true);
		Handles.color = Color.black;
		// Draw tiles in map:
		int index=0;
		foreach (var eb in mBlocks)
		{
			GUI.color = eb.index == 0 ? Color.cyan : eb.type == Block.Blocks.Finish ? Color.green : Color.white;

			Debug.Assert(eb.index == index++, "Index fail!");
			if (mSelection.Contains(eb))
				GUI.DrawTextureWithTexCoords(new Rect(eb.x * TileSize, eb.y * TileSize, TileSize, TileSize), mBackTex, rectDir);
			else
				DrawBlock(eb);
		}

		GUI.color = Color.white;

		// Draw jump lines on top:
		foreach (var eb in mBlocks)
		{
			if (eb.type == Block.Blocks.Jump)
			{
				float offset = TileSize*0.5f;
				var tgt = mBlocks[eb.index+1];
				var start = new Vector3(eb.x*TileSize+offset, eb.y*TileSize+offset, 0);
				var end = new Vector3(tgt.x*TileSize+offset, tgt.y*TileSize+offset, 0);
				var dir = (end-start).normalized * 8;
				Handles.DrawLine(start+dir, end);
			}
		}

		// Grid lines:
		Handles.color = Color.grey;
		for (int y = 0; y <= LevelHeight; y++)
			Handles.DrawLine(new Vector3(0, y * TileSize, 0), new Vector3(pixWidth, y * TileSize, 0));
		for (int x = 0; x <= LevelWidth; x++)
			Handles.DrawLine(new Vector3(x * TileSize, 0, 0), new Vector3(x * TileSize, pixHeight, 0));

		GUI.EndScrollView();
	}

	private void DrawBlock(EditorBlock eb)
	{
		var tex = eb.type == Block.Blocks.Blank ? mBackTex : mTileTex;
		var texRect = eb.type == Block.Blocks.Direction ? rectDir : eb.type == Block.Blocks.Jump ? rectJump : eb.type == Block.Blocks.Finish ? rectFinish : rectBlank;
		var matrix = GUI.matrix;
		if (eb.type == Block.Blocks.Direction || eb.type == Block.Blocks.Jump)
		{
			var nextBlock = mBlocks[eb.index+1];
			float angle = Mathf.Atan2(nextBlock.x-eb.x, nextBlock.y-eb.y) * Mathf.Rad2Deg;
			GUIUtility.RotateAroundPivot(90-angle, new Vector2(eb.x * TileSize + TileSize * 0.5f, eb.y * TileSize + TileSize * 0.5f));
		}
		GUI.DrawTextureWithTexCoords(new Rect(eb.x * TileSize, eb.y * TileSize, TileSize, TileSize), tex, texRect);
		GUI.matrix = matrix;
	}

	private void MoveSelection(int x, int y)
	{
		foreach (var b in mSelection)
		{
			b.x += x;
			b.y += y;
		}

		OnBlocksChanged();
	}

	private void SelectIsland(int x, int y)
	{
		var block = mBlocks.FirstOrDefault(b => b.x==x && b.y==y);
		if (block != null)
		{
			var island = GetIsland(block).ToList();
			if (mSelection.Contains(block))
			{
				foreach(var b in island)
					mSelection.Remove(b);
			}
			else
			{
				mSelection.AddRange(island);
			}
		}
		else
		{
			mSelection.Clear();
		}

		Repaint();
	}

	private void CheckAddTile(int x, int y)
	{
		if (mBlocks.Any(b => b.x==x && b.y==y))
			return;

		var adjBlock = mBlocks.FirstOrDefault(b => b.IsAdjacent(x,y));
		if (adjBlock == null)
		{
			AddBlock(x, y);
		}
		else
		{
			var island = GetIsland(adjBlock).OrderBy(b => b.index);
			var last = island.Last();
			if (last.IsAdjacent(x, y))
			{
				if (last.index != mBlocks.Count-1)
				{
					InsertBlock(last.index+1, x, y);
				}
				else
				{
					AddBlock(x,y);
				}
			}
			else
			{
				AddBlock(x,y);	// New island.
			}
		}
	}

	// Gets 'island' of blocks that includes the given block.
	private IEnumerable<EditorBlock> GetIsland(EditorBlock aBlock)
	{
		var island = new List<EditorBlock>(8);
		island.Add(aBlock);
		TraceIslandPath(aBlock, +1, island);
		TraceIslandPath(aBlock, -1, island);
		return island;
	}

	private void TraceIslandPath(EditorBlock aBlock, int aDelta, List<EditorBlock> aIsland)
	{
		int iNext = aBlock.index + aDelta;
		if (iNext < 0 || iNext >= mBlocks.Count)
			return;

		var next = mBlocks[iNext];
		if (next.IsAdjacent(aBlock.x, aBlock.y))
		{
			aIsland.Add(next);
			TraceIslandPath(next, aDelta, aIsland);
		}
	}

	private void AddBlock(int x, int y)
	{
		mBlocks.Add(new EditorBlock(mBlocks.Count, x, y));
		OnBlocksChanged();
	}

	private void InsertBlock(int aIndex, int x, int y)
	{
		mBlocks.Insert(aIndex, new EditorBlock(aIndex, x, y));
		for (int i=0 ; i<mBlocks.Count ; i++)
			mBlocks[i].index = i;
		OnBlocksChanged();
	}

	private void RemoveBlock(int aIndex)
	{
		mBlocks.RemoveAt(aIndex);
		for (int i=0 ; i<mBlocks.Count ; i++)
			mBlocks[i].index = i;
		OnBlocksChanged();
	}

	private void OnBlocksChanged()
	{
		mDirty = true;
		EvaluateLevel();
		Repaint();
	}

	private void Zoom(bool aIn)
	{
		TileSize += aIn ? 8 : -8;
		TileSize = Mathf.Clamp(TileSize, 16, 256);
		Repaint();
	}

	List<EditorBlock> mBlocks = new List<EditorBlock>(32);
	private void EvaluateLevel()
	{
		for (int i=0 ; i<mBlocks.Count ; i++)
		{
			var block = mBlocks[i];
			block.type = Block.Blocks.Blank;

			if (i == mBlocks.Count-1)
			{
				block.type = Block.Blocks.Finish;
				continue;
			}
			
			if (i==0)
				continue;	// Start block;

			var nextBlock = mBlocks[i+1];
			if (nextBlock.IsAdjacent(block.x, block.y))
			{
				var prevBlock = mBlocks[i-1];
				var dirPrevCur = (new Vector2(block.x, block.y) - new Vector2(prevBlock.x, prevBlock.y)).normalized;
				var dirCurNext = (new Vector2(nextBlock.x, nextBlock.y) - new Vector2(block.x, block.y)).normalized;
				if (dirPrevCur != dirCurNext)
				{
					block.type = Block.Blocks.Direction;
				}
			}
			else
			{
				block.type = Block.Blocks.Jump;
			}
		}
	}

	public class EditorBlock
	{
		public EditorBlock(int aIndex, int aX, int aY)
		{
			index = aIndex;
			x = aX;
			y = aY;
		}

		public int index;
		public int x;
		public int y;
		public Block.Blocks type;

		internal bool IsAdjacent(int aX, int aY)
		{
			int dx = aX-x;
			int dy = aY-y;
			return dx*dx + dy*dy == 1;
		}
	}

}


public class LevelGen
{
	private List<LvlEditor.EditorBlock> mBlocks;

	public List<LvlEditor.EditorBlock> Generate(string aSeed, float aDifficulty, float aLength, int aMaxWidth, int aMaxHeight)
	{
		mBlocks = new List<LvlEditor.EditorBlock>(128);

		int seed = 0;
		if (!int.TryParse(aSeed, out seed))
			seed = aSeed.GetHashCode();

		Debug.Log("Generate level, seed: " + seed.ToString());

		Random.InitState(seed);

		var dir = Direction.Up;
		int steps = (int)(aLength * 50) + Random.Range(10, 15);
		int x = 16;
		int y = 16;
		for (int s=0 ; s<steps ; s++)
		{
			int stepLen = Random.Range(2,6);
			for (int ss=0 ; ss<stepLen ; ss++)
			{
				if (IsOccupied(x, y))
				{
					var nu = FindNewPos(x, y);
					x = (int)nu.x;
					y = (int)nu.y;
				}
				
				mBlocks.Add(new LvlEditor.EditorBlock(mBlocks.Count, x, y));

				x += (int)Dirs[(int)dir].x;
				y += (int)Dirs[(int)dir].y;
				if (x<0) x += Random.Range(4,16);
				if (y<0) y += Random.Range(4,16);
			}

			// new dir:
			int d = (int)dir;
			d += (int)Mathf.Sign(Random.Range(-1.0f, 1.0f));
			if (d < 0) d += 4;
			if (d > 3) d -= 4;
			dir = (Direction)d;
		}

		//CentreMap(aMaxWidth, aMaxHeight);
		return mBlocks;
	}

	private Vector2 FindNewPos(int x, int y)
	{
		int range = 3;
		int nx=x, ny=y;
		do
		{
			for (int i=0 ; i<range*range ; i++)
			{
				nx = x + Random.Range(-range, range);
				ny = x + Random.Range(-range, range);
				if (!IsOccupied(nx, ny))
					break;
			}
			range++;
		} while (range < 20);

		return new Vector2(nx, ny);
	}

	private bool IsOccupied(int x, int y)
	{
		if (x<0 || y<0 || x>31 || y>31) return true;
		return mBlocks.FirstOrDefault(b => b.x==x && b.y==y) != null;
	}

	public void CentreMap(int aMaxWidth, int aMaxHeight)
	{
		if (mBlocks.Count < 2)
			return;
		var ox = mBlocks.OrderBy(b => b.x);
		int xMin = ox.First().x;
		int xMax = ox.Last().x;
		var oy = mBlocks.OrderBy(b => b.y);
		int yMin = ox.First().y;
		int yMax = ox.Last().y;
		int xRng = xMax-xMin;
		int yRng = yMax-yMin;
		int xOff = (aMaxWidth/2 - xRng/2) - xMin;
		int yOff = (aMaxHeight/2 - yRng/2) - yMin;
		foreach (var b in mBlocks)
		{
			b.x += xOff;
			b.y += yOff;
		}
	}

	enum Direction
	{
		Up,
		//UR,
		Right,
		//DR,
		Down,
		//DL,
		Left,
		//UL
	}

	static Vector2[] Dirs = new[]
	{
		new Vector2(0,1),
		//new Vector2(1,1),
		new Vector2(1,0),
		//new Vector2(1,-1),
		new Vector2(0,-1),
		//new Vector2(-1,-1),
		new Vector2(-1,0),
		//new Vector2(-1,1),
	};
}
